/*************************************************************
 *
 *  writeout.c - output subroutines for part3d
 *
 *  Copyright (C) 2000-2002  Mark J. Stock, mstock@umich.edu
 *
 *  This file is part of part3d.
 *
 *  part3d is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  part3d is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with part3d; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *********************************************************** */


#include "structs.h"

int write_particle_count(cell_ptr);
int write_pgm_plane(fileprop_ptr,cell_ptr,char*);
int write_pgm_density(fileprop_ptr,cell_ptr,field_ptr,char*);
int put_part_into_array(cell_ptr,cell_ptr,FLOAT**,int,int);
int write_rad_strand(sim_ptr,cell_ptr,char*);
int write_rad(sim_ptr,cell_ptr,char*);
int write_rad_cell(cell_ptr,FILE*,int);
int write_part(cell_ptr,char*);
int write_part_cell(cell_ptr,FILE*);


/*
 * write a heirarchical description of the particles in cells
 */
int write_particle_count(cell_ptr cell){

   int i,j,k;

   if (cell->level == 0) fprintf(stdout,"\n");

   if (cell->num > 0) {
      for (i=-1;i<cell->level;i++) fprintf(stdout,"  ");
      fprintf(stdout,"cell at level %d has %d particles\n",cell->level,cell->num);
   }

   if (cell->has_subcells)
      for (i=0;i<2;i++)
         for (j=0;j<2;j++)
            for (k=0;k<2;k++)
               write_particle_count(cell->s[i][j][k]);

   return(0);
}


/*
 * Write a PGM image of the points projected onto the xy-plane
 */
int write_pgm_plane(fileprop_ptr file,cell_ptr cell,char *filename){

   int do_fireworks = FALSE;
   int i,j;
   int nx = file->out_img_size;
   int ny = file->out_img_size;
   FLOAT scale = 1.0;
   FLOAT **array = (FLOAT **)malloc(nx * sizeof(FLOAT *));
   int printval;
   FILE *outfile;

   /* allocate space for 2D array */
   array[0] = (FLOAT *)malloc(nx * ny * sizeof(FLOAT));
   for (i=1; i<nx; i++)
      array[i] = array[0] + i * ny;

   /* zero the array */
   /* Not zeroing this array must use the pervious numbers, and this
      thing just adds the new locations to the old, looks like the
      burst of a firework! */
   if (do_fireworks) {
      /* do not clear memory */
   } else {
      for (j=ny-1; j>=0; j--) {
         for (i=0; i<nx; i++) {
            array[i][j] = 0.0;
         }
      }
   }

   /* fill in the array */
   put_part_into_array(cell,cell,array,nx,ny);

   /* open file for writing */
   outfile = fopen(filename,"w");
   if (outfile==NULL) {
      fprintf(stderr,"Could not open output file %s\n",filename);
      fflush(stderr);
      exit(0);
   }

   /* plot an z-plane */
   fprintf(outfile,"P2\n%d %d\n%d\n",nx,ny,255);
   for (j=ny-1; j>=0; j--) {
      for (i=0; i<nx; i++) {
         printval = (int)(256*array[i][j]*scale);
         if (printval<0) printval = 0;
         if (printval>255) printval = 255;
         fprintf(outfile,"%d\n",printval);
      }
   }

   /* close file */
   fclose(outfile);

   /* free memory from 2D array! */
   free(array[0]);
   free(array);

   /* if all went well, return zero */
   return(0);
}


/*
 * Write a PGM image of the density field
 */
int write_pgm_density(fileprop_ptr file,cell_ptr cell,field_ptr ff,char *filename){

   int do_middle_slice = FALSE;
   int i,j,k;
   int kindex = ff->n[1]/2;
   int nx = ff->n[0];
   int ny = ff->n[2];
   FLOAT scale = 2.0;
   FLOAT **array = (FLOAT **)malloc(nx * sizeof(FLOAT *));
   int printval;
   FILE *outfile;

   /* allocate space for 2D array */
   array[0] = (FLOAT *)malloc(nx * ny * sizeof(FLOAT));
   for (i=1; i<nx; i++)
      array[i] = array[0] + i * ny;

   /* zero the array */
   for (j=ny-1; j>=0; j--)
      for (i=0; i<nx; i++)
         array[i][j] = 0.0;

   /* fill the array */
   for (i=0; i<nx; i++)
      for (j=0; j<ny; j++) {
         if (do_middle_slice) array[i][j] = ff->rho[i][kindex][j];
         else {
            for (k=0; k<ff->n[1]; k++)
               array[i][j] += ff->rho[i][k][j];
            array[i][j] *= 0.1;
         }
      }


   /* open file for writing */
   outfile = fopen(filename,"w");
   if (outfile==NULL) {
      fprintf(stderr,"Could not open output file %s\n",filename);
      fflush(stderr);
      exit(0);
   }

   /* plot a y-plane */
   fprintf(outfile,"P2\n%d %d\n%d\n",nx,ny,255);
   for (j=ny-1; j>=0; j--) {
      for (i=0; i<nx; i++) {
         printval = (int)(256.0*array[i][j]*scale);
         if (printval<0) printval = 0;
         if (printval>255) printval = 255;
         fprintf(outfile,"%d\n",printval);
      }
   }

   /* close file */
   fclose(outfile);

   /* free memory from 2D array! */
   free(array[0]);
   free(array);

   /* if all went well, return zero */
   return(0);
}


/*
 * Put all of the particles in this cell onto the array
 */
int put_part_into_array(cell_ptr top,cell_ptr cell,FLOAT** array,int nx,int ny){

   int i,j,k;
   int xdim,ydim;
   FLOAT xstart,xsize,ystart,ysize,mult;
   particle_ptr curr;

   xdim = 0;	/* use x-values for x-dim of image */
   ydim = 2;	/* use z-values for y-dim of image */

   if (cell->has_subcells) {
      for (i=0; i<2; i++)
         for (j=0; j<2; j++)
            for (k=0; k<2; k++)
               /* put_part_into_array(top,cell,&array[0][0],nx,ny); */
               put_part_into_array(top,cell->s[i][j][k],array,nx,ny);
   } else {
      /* run through the list of elems and place each on the grid */
      // mult = nx*ny*0.05;
      mult = 255.0*sqrt(nx*ny);
      xstart = top->min[xdim];
      xsize = top->max[xdim]-xstart;
      ystart = top->min[ydim];
      ysize = top->max[ydim]-ystart;
      curr = cell->first;
      while (curr) {
         i = (int)(nx*(curr->x[0][xdim]-xstart)/xsize);
         j = (int)(ny*(curr->x[0][ydim]-xstart)/xsize);
         if (i>-1 && j>-1 && i<nx && j<ny)
            array[i][j] += mult * curr->rad;
            // array[i][j] += mult * curr->mass;
         curr = curr->next;
      }
   }
   return(0);
}


/*
 * Write a Radiance description of the particles and cell edges
 */
int write_rad(sim_ptr sim,cell_ptr top,char *filename){

   int type = 1;
   FILE *outfile;

   /* open file for writing */
   // outfile = fopen(filename,"w");
   outfile = fopen(filename,"a");
   if (outfile==NULL) {
      fprintf(stderr,"Could not open output file %s\n",filename);
      fflush(stderr);
      exit(0);
   }

   fprintf(outfile,"# Radiance scene description of grav3d run\n");
   if (type == 0) {
      /* write all cell edges, and particles as lights */
      fprintf(outfile,"void plastic edgec 0 0 5 0.2 0.2 0.2 0.0 0.0\n");
      fprintf(outfile,"void light partc 0 0 3 1000 1000 1000\n");
   } else if (type == 1) {
      /* write stars as plastic objects, and a light source overhead */
      /* fprintf(outfile,"void plastic wallc 0 0 5  0.2 0.2 0.2 0.0 0.0\n"); */
      /* fprintf(outfile,"wallc polygon floor 0 0 12  0 0 0  1 0 0  1 1 0  0 1 0\n"); */
      fprintf(outfile,"void light lightc 0 0 3  10 10 10\n");
      fprintf(outfile,"lightc polygon l1 0 0 12  0 0 2  0 1 2  1 1 2  1 0 2\n");
      fprintf(outfile,"void plastic partc 0 0 5  0.5 0.5 0.5 0.0 0.0\n");
      if (sim->bdry[2][0] == WALL) fprintf(outfile,"partc polygon floor 0 0 12 %g %g %g %g %g %g %g %g %g %g %g %g\n", top->min[0],top->min[1],top->min[2],top->max[0],top->min[1],top->min[2], top->max[0],top->max[1],top->min[2],top->min[0],top->max[1],top->min[2]);
   } else if (type == 2) {
      /* write stars as glow objects */
      fprintf(outfile,"void glow partc 0 0 4  1.0 1.0 1.0 0.0\n");
   }

   write_rad_cell(top,outfile,type);

   /* close file */
   fclose(outfile);

   /* if all went well, return zero */
   return(0);
}


/*
 * Write a Radiance description of just this cell
 */
int write_rad_cell(cell_ptr c,FILE* outfile,int type){

   int i,j,k,cnt;
   FLOAT rad;
   particle_ptr curr;

   fprintf(outfile,"\n# cell at level %d\n",c->level);

   if (c->has_subcells) {
      /* recurse for the subcells */
      for (i=0;i<2;i++)
         for (j=0;j<2;j++)
            for (k=0;k<2;k++)
               write_rad_cell(c->s[i][j][k],outfile,type);
   } else {
      if (type == 0) {
         /* write out the edges */
         rad = 0.001;
         fprintf(outfile,"edgec cylinder e1 0 0 7 %g %g %g %g %g %g %g\n",c->min[0],c->min[1],c->min[2],c->max[0],c->min[1],c->min[2],rad);
         fprintf(outfile,"edgec cylinder e2 0 0 7 %g %g %g %g %g %g %g\n",c->min[0],c->min[1],c->max[2],c->max[0],c->min[1],c->max[2],rad);
         fprintf(outfile,"edgec cylinder e3 0 0 7 %g %g %g %g %g %g %g\n",c->min[0],c->max[1],c->min[2],c->max[0],c->max[1],c->min[2],rad);
         fprintf(outfile,"edgec cylinder e4 0 0 7 %g %g %g %g %g %g %g\n",c->min[0],c->max[1],c->max[2],c->max[0],c->max[1],c->max[2],rad);
         fprintf(outfile,"edgec cylinder e5 0 0 7 %g %g %g %g %g %g %g\n",c->min[0],c->min[1],c->min[2],c->min[0],c->max[1],c->min[2],rad);
         fprintf(outfile,"edgec cylinder e6 0 0 7 %g %g %g %g %g %g %g\n",c->min[0],c->min[1],c->max[2],c->min[0],c->max[1],c->max[2],rad);
         fprintf(outfile,"edgec cylinder e7 0 0 7 %g %g %g %g %g %g %g\n",c->max[0],c->min[1],c->min[2],c->max[0],c->max[1],c->min[2],rad);
         fprintf(outfile,"edgec cylinder e8 0 0 7 %g %g %g %g %g %g %g\n",c->max[0],c->min[1],c->max[2],c->max[0],c->max[1],c->max[2],rad);
         fprintf(outfile,"edgec cylinder e9 0 0 7 %g %g %g %g %g %g %g\n",c->min[0],c->min[1],c->min[2],c->min[0],c->min[1],c->max[2],rad);
         fprintf(outfile,"edgec cylinder e10 0 0 7 %g %g %g %g %g %g %g\n",c->min[0],c->max[1],c->min[2],c->min[0],c->max[1],c->max[2],rad);
         fprintf(outfile,"edgec cylinder e11 0 0 7 %g %g %g %g %g %g %g\n",c->max[0],c->min[1],c->min[2],c->max[0],c->min[1],c->max[2],rad);
         fprintf(outfile,"edgec cylinder e12 0 0 7 %g %g %g %g %g %g %g\n",c->max[0],c->max[1],c->min[2],c->max[0],c->max[1],c->max[2],rad);
         fflush(outfile);
      } else if (type >= 1) {
         /* don't write out the edges */
      }

      /* write out the particles */
      curr = c->first;
      cnt = 0;
      while (curr) {
         /* if (type == 0) rad = 0.001*sqrt(curr->mass); */
         /* else if (type == 1) rad = 0.5*sqrt(curr->mass); */
         /* else if (type >= 1) rad = pow(curr->mass,1./3.)/50.; */
         /* rad = pow(curr->mass,1./3.)/50.; */
         // if the particle is not part of a strand, print it
         if (!curr->parent) {
            fprintf(outfile,"partc sphere p%d 0 0 4 %g %g %g %g\n",cnt,curr->x[0][0],curr->x[0][1],curr->x[0][2],curr->rad);
            // fflush(outfile);
            cnt++;
            curr = curr->next;
         }
      }
   }

   /* if all went well, return zero */
   return(0);
}


/*
 * Write a Radiance description of the strands
 */
int write_rad_strand(sim_ptr sim,cell_ptr top,char *filename){

   int write_genworm = TRUE;
   int write_genworm_endcaps = TRUE;
   int i,j,cnt,numseg;
   FLOAT dir[2][3],ddss[2][3],len,radcurv[2];
   FILE *outfile;
   particle_ptr this,next,nextnext,last;

   /* open file for writing */
   // outfile = fopen(filename,"w");
   outfile = fopen(filename,"a");
   if (outfile==NULL) {
      fprintf(stderr,"Could not open output file %s\n",filename);
      fflush(stderr);
      exit(0);
   }

   fprintf(outfile,"# Radiance scene description of grav3d run\n");
   // fprintf(outfile,"void light lightc 0 0 3  10 10 10\n");
   // fprintf(outfile,"lightc polygon l1 0 0 12  0 0 2  0 1 2  1 1 2  1 0 2\n");
   // fprintf(outfile,"void plastic partc 0 0 5  0.5 0.5 0.5 0.0 0.0\n");
   // if (sim->bdry[2][0] == WALL) fprintf(outfile,"partc polygon floor 0 0 12 %g %g %g %g %g %g %g %g %g %g %g %g\n", top->min[0],top->min[1],top->min[2],top->max[0],top->min[1],top->min[2], top->max[0],top->max[1],top->min[2],top->min[0],top->max[1],top->min[2]);

   for (i=0; i<sim->num_strands; i++) {
      fprintf(outfile,"\n# strand %d\n",i);
      // this = sim->strand_start[i];
      this = sim->strand[i].start;
      cnt = 0;
      while (this) {
         next = this->s_next;
         if (next) nextnext = next->s_next;
         last = this->s_last;

         if (write_genworm && next) {

            // the directions are merely the lines joining i-1 and i+1
            if (last) {
               for (j=0;j<3;j++) dir[0][j] = next->x[0][j]-last->x[0][j];
               for (j=0;j<3;j++) ddss[0][j] = next->x[0][j]-2*this->x[0][j]+last->x[0][j];
            }
            if (nextnext) {
               for (j=0;j<3;j++) dir[1][j] = nextnext->x[0][j]-this->x[0][j];
               for (j=0;j<3;j++) ddss[1][j] = nextnext->x[0][j]-2*next->x[0][j]+this->x[0][j];
            }

            // but, if this is the first particle/segment, compute start direction
            if (!last) {
               for (j=0;j<3;j++) dir[0][j] = -3*this->x[0][j] + 4*next->x[0][j] - nextnext->x[0][j];
               for (j=0;j<3;j++) ddss[0][j] = ddss[1][j];
            }

            // or, if this is the last particle/segment, compute end direction
            if (!nextnext) {
               for (j=0;j<3;j++) dir[1][j] = 3*next->x[0][j] - 4*this->x[0][j] + last->x[0][j];
               for (j=0;j<3;j++) ddss[1][j] = ddss[0][j];
            }

            // normalize both
            len = sqrt(dir[0][0]*dir[0][0]+dir[0][1]*dir[0][1]+dir[0][2]*dir[0][2]);
            for (j=0;j<3;j++) dir[0][j] /= len;
            len = sqrt(dir[1][0]*dir[1][0]+dir[1][1]*dir[1][1]+dir[1][2]*dir[1][2]);
            for (j=0;j<3;j++) dir[1][j] /= len;
            // find the segment length, and scale each direction by it
            // (using the halflength still shows the segment bounds)
            len = sqrt(pow(next->x[0][0]-this->x[0][0],2)+pow(next->x[0][1]-this->x[0][1],2)+pow(next->x[0][2]-this->x[0][2],2));
            for (j=0;j<3;j++) dir[0][j] *= len;
            for (j=0;j<3;j++) dir[1][j] *= len;

            fprintf(outfile,"!genworm partc w_%d_%d ",i,cnt);
            fprintf(outfile,"'hermite (%g,%g,%g,%g,t)' ",this->x[0][0],next->x[0][0],dir[0][0],dir[1][0]);
            fprintf(outfile,"'hermite (%g,%g,%g,%g,t)' ",this->x[0][1],next->x[0][1],dir[0][1],dir[1][1]);
            fprintf(outfile,"'hermite (%g,%g,%g,%g,t)' ",this->x[0][2],next->x[0][2],dir[0][2],dir[1][2]);

            // find the radius of curvature (at both ends)

            for (j=0;j<3;j++) ddss[0][j] /= len*len;
            for (j=0;j<3;j++) ddss[1][j] /= len*len;
            // fprintf(stdout,"\n%g %g %g\n",ddss[0][0],ddss[0][1],ddss[0][2]);
            // fprintf(stdout,"%g %g %g\n",ddss[1][0],ddss[1][1],ddss[1][2]);

            radcurv[0] = 0.0;
            for (j=0;j<3;j++) radcurv[0] += ddss[0][j]*ddss[0][j];
            radcurv[0] = 1.0/sqrt(radcurv[0]);

            radcurv[1] = 0.0;
            for (j=0;j<3;j++) radcurv[1] += ddss[1][j]*ddss[1][j];
            radcurv[1] = 1.0/sqrt(radcurv[1]);

            len = sqrt(radcurv[0]*radcurv[1]);

            // write the worm radius and number of segments
            if (sim->num_strands > 100) numseg = (int)(1.5 + 10.0*atan(this->s_rad/len));
            else if (sim->num_strands > 10) numseg = (int)(1.5 + 20.0*atan(this->s_rad/len));
            else numseg = (int)(1.5 + 30.0*atan(this->s_rad/len));
            // fprintf(stdout,"%g %d\n",len,numseg);
            if (numseg > 15) numseg = 15;
            fprintf(outfile,"%g %d\n",this->rad,numseg);

            // genworm ends in a sphere, we want a cylinder
            if (write_genworm_endcaps) {

               // if this is the first particle
               if (!last) {

                  // compute the direction vector
                  len = sqrt(dir[0][0]*dir[0][0]+dir[0][1]*dir[0][1]+dir[0][2]*dir[0][2]);
                  // normalize it, and reverse the direction
                  for (j=0;j<3;j++) dir[0][j] /= -1.0*len;

                  // use the other dir, dir[1][0:2] as the end point
                  for (j=0;j<3;j++) dir[1][j] = this->x[0][j]+this->rad*dir[0][j];

                  // write the cylinder
                  fprintf(outfile,"partc cylinder c_%d_%d 0 0 7 ",i,cnt);
                  fprintf(outfile,"%g %g %g  %g %g %g  %g\n",this->x[0][0],this->x[0][1],this->x[0][2],dir[1][0],dir[1][1],dir[1][2],this->rad);

                  // write the ring
                  fprintf(outfile,"partc ring r_%d_%d 0 0 8 ",i,cnt);
                  fprintf(outfile,"%g %g %g  %g %g %g  0.0 %g\n",dir[1][0],dir[1][1],dir[1][2],dir[0][0],dir[0][1],dir[0][2],this->rad);

               // or, if this is the last particle
               } else if (!nextnext) {

                  // compute the direction vector at "next"
                  len = sqrt(dir[1][0]*dir[1][0]+dir[1][1]*dir[1][1]+dir[1][2]*dir[1][2]);
                  // normalize it, do not reverse the direction
                  for (j=0;j<3;j++) dir[1][j] /= len;

                  // use the other dir, dir[0][0:2] as the end point
                  for (j=0;j<3;j++) dir[0][j] = next->x[0][j]+next->rad*dir[1][j];

                  // write the cylinder
                  fprintf(outfile,"partc cylinder c_%d_%d 0 0 7 ",i,cnt);
                  fprintf(outfile,"%g %g %g  %g %g %g  %g\n",next->x[0][0],next->x[0][1],next->x[0][2],dir[0][0],dir[0][1],dir[0][2],next->rad);

                  // write the ring
                  fprintf(outfile,"partc ring r_%d_%d 0 0 8 ",i,cnt);
                  fprintf(outfile,"%g %g %g  %g %g %g  0.0 %g\n",dir[0][0],dir[0][1],dir[0][2],dir[1][0],dir[1][1],dir[1][2],next->rad);

               }
            }

         } else if (!write_genworm) {

            // write using raw segments, capped with rings

            // if this is the last particle, cap the strand
            if (!next) fprintf(outfile,"partc ring re_%d\n0 0 8 %g %g %g %g %g %g %g %g\n",i,this->x[0][0],this->x[0][1],this->x[0][2],this->x[0][0]-this->s_last->x[0][0],this->x[0][1]-this->s_last->x[0][1],this->x[0][2]-this->s_last->x[0][2],0.0,this->rad);

            // if this is the first particle, cap the strand
            if (!this->s_last) fprintf(outfile,"partc ring rs_%d\n0 0 8 %g %g %g %g %g %g %g %g\n",i,this->x[0][0],this->x[0][1],this->x[0][2],this->x[0][0]-next->x[0][0],this->x[0][1]-next->x[0][1],this->x[0][2]-next->x[0][2],0.0,this->rad);

            // write the sphere unless it's an end
            if (next && this->s_last) fprintf(outfile,"partc sphere s_%d_%d\n0 0 4 %g %g %g %g\n",i,cnt,this->x[0][0],this->x[0][1],this->x[0][2],this->rad);

            // write the segment only if this isn't the last one
            if (next) fprintf(outfile,"partc cylinder c_%d_%d\n0 0 7 %g %g %g %g %g %g %g\n",i,cnt,this->x[0][0],this->x[0][1],this->x[0][2],next->x[0][0],next->x[0][1],next->x[0][2],this->rad);

         }

         cnt++;
         this = this->s_next;
         fflush(outfile);
      }
   }

   /* close file */
   fclose(outfile);

   // wait until a keystroke
   // getchar();

   /* if all went well, return zero */
   return(0);
}


/*
 * Write a re-readable native particle3d description of the particles
 */
int write_part(cell_ptr top,char *filename){

   FILE *outfile;

   /* open file for writing */
   outfile = fopen(filename,"w");
   if (outfile==NULL) {
      fprintf(stderr,"Could not open output file %s\n",filename);
      fflush(stderr);
      exit(0);
   }

   fprintf(outfile,"# part3d scene description of grav3d run\n");

   write_part_cell(top,outfile);

   /* close file */
   fclose(outfile);

   /* if all went well, return zero */
   return(0);
}


/*
 * Write a native particle3d description of just this cell
 */
int write_part_cell(cell_ptr c,FILE* outfile){

   int i,j,k,cnt;
   FLOAT rad;
   particle_ptr curr;

   /* fprintf(outfile,"\n# cell at level %d\n",c->level); */

   if (c->has_subcells) {
      /* recurse for the subcells */
      for (i=0;i<2;i++)
         for (j=0;j<2;j++)
            for (k=0;k<2;k++)
               write_part_cell(c->s[i][j][k],outfile);
   } else {
      /* write out the particles */
      curr = c->first;
      cnt = 0;
      while (curr) {
         fprintf(outfile,"%g %g %g %g %g %g %g\n",curr->x[0][0],curr->x[0][1],curr->x[0][2],curr->rad,curr->u[0][0],curr->u[0][1],curr->u[0][2]);
         /* fflush(outfile); */
         cnt++;
         curr = curr->next;
      }
   }

   /* if all went well, return zero */
   return(0);
}

