/*************************************************************
 *
 *  setup.c - subroutines for setting up simulations
 *
 *  Copyright (C) 2000-2003  Mark J. Stock, mstock@umich.edu
 *
 *  This file is part of part3d.
 *
 *  part3d is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  part3d is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with part3d; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *********************************************************** */


#include "structs.h"
#include <time.h>

int initialize_system(fileprop_ptr,sim_ptr,cell_ptr);
int add_particles_from_file(sim_ptr,cell_ptr,int,char);
int add_box_of_particles(sim_ptr,cell_ptr,int,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT);
int add_box_of_ni_particles(sim_ptr,cell_ptr,int,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT);
int add_sphere_of_particles(sim_ptr,cell_ptr,int,FLOAT,FLOAT,FLOAT,FLOAT);
int add_sphere_of_ni_particles(sim_ptr,cell_ptr,int,FLOAT,FLOAT,FLOAT,FLOAT);
int add_strand_of_particles(sim_ptr,cell_ptr,int,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT);
int is_this_space_open(cell_ptr,FLOAT,FLOAT,FLOAT,FLOAT);
int is_this_space_open2(cell_ptr,FLOAT,FLOAT,FLOAT,FLOAT);
int add_galaxy(sim_ptr,cell_ptr,int,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT);
particle_ptr new_particle(int,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT,FLOAT);
particle_ptr new_stationary_particle(int,FLOAT,FLOAT,FLOAT,FLOAT);
int read_input_file(fileprop_ptr,sim_ptr,cell_ptr);
int parse_args(int,char**,fileprop_ptr);
int Usage(char[80],int);
FLOAT*** allocate_3d_array_F(int,int,int);
FLOAT** allocate_2d_array_F(int,int);


/*
 *  Initialize the system - make static and free element structures, etc.
 */
int initialize_system(fileprop_ptr file,sim_ptr sim,cell_ptr top) {

   int i,j,k,ret_val,num_parts;
   FLOAT xpos,ypos,zpos,xvel,yvel,zvel,distsq,mpp;
   FLOAT r,rcubed,orbvel,M;
   particle_ptr newpart;

   // ------------------------------------------------------------------------------

   /* set default values for variables */
   strcpy(file->out_fn_root,"output");
   file->write_pgm = FALSE;
   file->write_gif = TRUE;
   file->write_png = FALSE;
   file->write_dot = FALSE;
   file->write_rad = FALSE;
   file->write_part = FALSE;

   sim->step = 0;
   sim->maxstep = 999999999;			/* make this sufficiently high */
   sim->start_time = 0.0;
   sim->end_time = 999999.0;			/* this too */
   sim->dt = 0.001;
   sim->max_dt = 0.001;
   sim->output_dt = 0.01;
   file->out_img_size = 256;
   sim->max_levels = 10;
   sim->max_parts_in_cell = 40;

   sim->time = sim->start_time;
   sim->next_output_time = sim->start_time;
   sim->next_output_index = 0;

   sim->use_self_grav = TRUE;
   sim->G = 0.0001;
   sim->delta = 0.001;

   sim->use_uniform_grav = FALSE;
   sim->g[0] = 0.0;
   sim->g[1] = 0.0;
   sim->g[2] = -9.8;

   sim->use_contact = FALSE;
   sim->rk = 1.0e+9;
   sim->rn = 4;
   sim->rc = 1.0e-6;
   sim->ak = 1.0;
   sim->an = 4;

   sim->bdry[0][0] = WALL;
   sim->bdry[0][1] = WALL;
   sim->bdry[1][0] = WALL;
   sim->bdry[1][1] = WALL;
   sim->bdry[2][0] = WALL;
   sim->bdry[2][1] = WALL;

   sim->particle_cnt = 1000;
   sim->new_part_rad = 0.005;
   sim->new_part_mass = 1.0;

   sim->num_blocks = 0;
   sim->num_spheres = 0;
   sim->num_indiv_parts = 0;
   sim->num_read_part = 0;
   sim->num_read_stat = 0;
   sim->num_strands = 0;
   sim->use_density_field = FALSE;
   sim->construct_dens_surface = FALSE;


   /* assign default values for the top-level cell */
   top->level = 0;
   top->has_subcells = FALSE;
   top->first = NULL;
   for (i=0;i<2;i++)
      for (j=0;j<2;j++)
         for (k=0;k<2;k++)
            top->s[i][j][k] = NULL;
   top->min[0] = 0.0;
   top->min[1] = 0.0;
   top->min[2] = 0.0;
   top->max[0] = 1.0;
   top->max[1] = 1.0;
   top->max[2] = 1.0;
   top->num = 0;
   top->mass = 0.0;

   // ------------------------------------------------------------------------------

   /* now, if the file listed on the command line exists, read it */
   /* ret_val is 0 if file is OK and read, 1 if not */
   ret_val = read_input_file(file,sim,top);

   if (ret_val == 1) {
      fprintf(stdout,"ERROR upon reading imput file, please check syntax\n");
      exit(1);
   }

   // ------------------------------------------------------------------------------

   // I think we only need these for the self-gravitation thing
   for (i=0;i<3;i++) top->mid[i] = (top->min[i]+top->max[i])/2.0;
   for (i=0;i<3;i++) top->cm[i] = (top->min[i]+top->max[i])/2.0;

   // ------------------------------------------------------------------------------
   // now, place particles

   for (i=0; i<sim->num_read_part; i++)
      add_particles_from_file(sim,top,i,FALSE);

   for (i=0; i<sim->num_read_stat; i++)
      add_particles_from_file(sim,top,i,TRUE);

   for (i=0; i<sim->num_strands; i++) {
      // fprintf(stdout,"sending %d %lf %lf %lf\n",i,sim->block[i][0],sim->block[i][1],sim->block[i][2]);
      add_strand_of_particles(sim,top,i,sim->block[i][0],
         sim->block[i][1],sim->block[i][2],sim->block[i][3],
         sim->block[i][4],sim->block[i][5],sim->block[i][6],
         sim->block[i][7],sim->block[i][8],sim->block[i][9],
         sim->block[i][10],sim->block[i][11]);
   }

   // then, place specific particles
   for (i=0; i<sim->num_indiv_parts; i++) {
      newpart = new_particle(i,sim->indiv_part[i][0],sim->new_part_rad,
                   sim->indiv_part[i][1],sim->indiv_part[i][2],sim->indiv_part[i][3],
                   sim->indiv_part[i][4],sim->indiv_part[i][5],sim->indiv_part[i][6]);
      add_particle_to_cell(sim,newpart,top);
   }

   // lastly, place particles to fill cubical or spherical volumes
   for (i=0; i<sim->num_blocks; i++)
      add_box_of_particles(sim,top,(int)(sim->block[i][0]),
         sim->block[i][1],sim->block[i][2],sim->block[i][3],
         sim->block[i][4],sim->block[i][5],sim->block[i][6]);

   // add a box of non-intersecting particles!
   for (i=0; i<sim->num_niblocks; i++)
      add_box_of_ni_particles(sim,top,(int)(sim->niblock[i][0]),
         sim->niblock[i][1],sim->niblock[i][2],sim->niblock[i][3],
         sim->niblock[i][4],sim->niblock[i][5],sim->niblock[i][6]);

   // add a sphere of randomly-placed particles
   for (i=0; i<sim->num_spheres; i++)
      add_sphere_of_particles(sim,top,(int)(sim->sphere[i][0]),
         sim->sphere[i][1],sim->sphere[i][2],sim->sphere[i][3],
         sim->sphere[i][4]);

   // add a sphere of non-intersecting particles
   for (i=0; i<sim->num_nispheres; i++)
      add_sphere_of_ni_particles(sim,top,(int)(sim->nisphere[i][0]),
         sim->nisphere[i][1],sim->nisphere[i][2],sim->nisphere[i][3],
         sim->nisphere[i][4]);

   // ------------------------------------------------------------------------------

   if (sim->use_density_field) {
      // set the cell sizes
      for (i=0; i<3; i++) sim->ff2->d[i] = (top->max[i]-top->min[i])/sim->ff2->n[i];

      // allocate memory for the array
      // ff->rho = allocate_3d_array_F(ff->n[0],ff->n[1],ff->n[2]);
      sim->ff2->rho = allocate_2d_array_F(sim->ff2->n[0],sim->ff2->n[2]);

      // zero the array
      for (i=0;i<sim->ff2->n[0];i++)
         for (j=0;j<sim->ff2->n[1];j++)
            sim->ff2->rho[i][j] = 0.0;
            // for (k=0;k<sim->ff2->n[2];k++)
               // ff->rho[i][j][k] = 0.0;
   }


   if (1==0) {
      /* create a lump of randomly-places masses */
      /* now they're in a sphere, and with a uniform spherical velocity profile */
      /* make this a subroutine called add_galaxy(x,y,z,xn,yn,zn,count); */
      num_parts = sim->particle_cnt;
      mpp = 1.0/num_parts;
      mpp = sim->new_part_mass;
      for (i=0;i<num_parts;i++) {
         xpos = top->mid[0] + (top->max[0]-top->min[0])*(rand()/(RAND_MAX+1.0)-0.5);
         ypos = top->mid[1] + (top->max[1]-top->min[1])*(rand()/(RAND_MAX+1.0)-0.5);
         zpos = top->mid[2] + 0.2*(top->max[2]-top->min[2])*(rand()/(RAND_MAX+1.0)-0.5);
         distsq = ((xpos-0.5)*(xpos-0.5)+(ypos-0.5)*(ypos-0.5)+5*(zpos-0.5)*5*(zpos-0.5));
         if (distsq < 0.25) {
            if (1==0) {
               /* create velocities randomly */
               xvel = 0.1-0.2*rand()/(RAND_MAX+1.0);
               yvel = 0.1-0.2*rand()/(RAND_MAX+1.0);
               zvel = 0.1-0.2*rand()/(RAND_MAX+1.0);
               distsq = ((xvel-0.1)*(xvel-0.1)+(yvel-0.1)*(yvel-0.1)+(zvel-0.1)*(zvel-0.1));
               if (distsq < 0.01) {
                  /* newpart = new_particle(i,mpp,sim->new_part_rad,xpos,ypos,zpos,xvel,yvel,zvel); */
                  newpart = new_particle(i,mpp,sim->new_part_rad,xpos,ypos,zpos,0.0,0.0,0.0);
                  add_particle_to_cell(sim,newpart,top);
               } else {
                  i--;
               }
            } else {
               /* velocities determined by orbit speed */
               distsq = ((xpos-0.5)*(xpos-0.5)+(ypos-0.5)*(ypos-0.5));
               r = sqrt(distsq);
               rcubed = r*r*r;
               M = mpp*num_parts*(r/0.5)*(r/0.5);
               orbvel = sqrt(sim->G*M/rcubed);
               xvel = -2.0*(ypos-0.5)*orbvel*15;
               yvel = 2.0*(xpos-0.5)*orbvel*15;
               zvel = 0.0;
               /* newpart = new_particle(i,mpp,sim->new_part_rad,xpos,ypos,zpos,xvel,yvel,zvel); */
               newpart = new_particle(i,mpp,sim->new_part_rad,xpos,ypos,zpos,0.0,0.0,0.0);
               add_particle_to_cell(sim,newpart,top);
            }
         } else {
           i--;
         }
      }
   }

   fprintf(stdout,"\nStarting with %d particles",top->num);

   // and the output file (just dots)
   for (i=0; i<3; i++) file->out->d[i] = (top->max[i]-top->min[i])/file->out->n[i];
   file->out->rho = allocate_2d_array_F(file->out_img_size,file->out_img_size);

   /* return 0, everything OK */
   return(0);
}


/*
 *  Read particle locations from a file, include them in the simulation
 *  Do this for both static and non-static particles
 */
int add_particles_from_file(sim_ptr sim,cell_ptr top,int i,char is_stat) {

   FLOAT x,y,z,rad,u,v,w;
   char token[7][16];
   char twochar[2];
   char sbuf[512];
   int cnt = 0;
   particle_ptr newpart;
   FILE *infile;

   /* open file for reading */
   if (is_stat) {
      infile = fopen(sim->read_stat[i],"r");
      if (infile==NULL) {
         fprintf(stderr,"Could not open particle file %s\n",sim->read_stat[i]);
         fflush(stderr);
         return(1);
      }
      fprintf(stdout,"Opening file %s\n",sim->read_stat[i]);
      fflush(stdout);
   } else {
      infile = fopen(sim->read_part[i],"r");
      if (infile==NULL) {
         fprintf(stderr,"Could not open particle file %s\n",sim->read_part[i]);
         fflush(stderr);
         return(1);
      }
      fprintf(stdout,"Opening file %s\n",sim->read_part[i]);
      fflush(stdout);
   }

   /* read a line from the input file */
   while (fscanf(infile,"%[^\n]",sbuf) != EOF) {

      /* fprintf(stdout,"%s\n",sbuf); */

      /* grab the line */
      sscanf(sbuf,"%s %s %s %s %s %s %s",token[0],token[1],token[2],token[3],token[4],token[5],token[6]);
      /* fprintf(stdout,"   first token is %s\n",token[0]); */

      if (token[0][0] == '#') {
         /* read a comment line, or some other descriptor */
         fscanf(infile,"%[\n]",twochar);    /* read up to newline */
         /* fprintf(stdout,"%s\n",sbuf);        /* write comment */

      } else {
         /* read the particle parameters */
         /* fprintf(stdout,"   second token is %s\n",token[1]); */

         x = atof(token[0]);
         y = atof(token[1]);
         z = atof(token[2]);
         rad = atof(token[3]);
         u = atof(token[4]);
         v = atof(token[5]);
         w = atof(token[6]);

         if (is_stat) {
            x += sim->stat_xform[i][0];
            y += sim->stat_xform[i][1];
            z += sim->stat_xform[i][2];
            newpart = new_particle(cnt,sim->new_part_mass,rad,x,y,z,0.0,0.0,0.0);
         } else {
            x += sim->part_xform[i][0];
            y += sim->part_xform[i][1];
            z += sim->part_xform[i][2];
            u += sim->part_xform[i][3];
            v += sim->part_xform[i][4];
            w += sim->part_xform[i][5];
            newpart = new_particle(cnt,sim->new_part_mass,rad,x,y,z,u,v,w);
         }

         add_particle_to_cell(sim,newpart,top);

         cnt++;
         if (cnt/1000 != (cnt+1)/1000) {
            fprintf(stdout,".");
            fflush(stdout);
         }

         fscanf(infile,"%[\n]",twochar);    /* read newline */
      }
   }

   fprintf(stdout,"\n");
   fprintf(stdout,"Placed %d particles\n",cnt);
   fflush(stdout);

   fclose(infile);

   /* ret_val is 0 is file is OK and read, 1 if not */
   return(0);
}


/*
 *  Create a cube/rectangle of particles (can be intersecting)
 */
int add_box_of_particles(sim_ptr sim,cell_ptr top,int cnt,FLOAT xs,FLOAT xf,FLOAT ys,FLOAT yf,FLOAT zs,FLOAT zf) {

   int i,keep_trying,ntried,isitfree,quit_altogether;
   FLOAT xpos,ypos,zpos;
   FLOAT sx,sy,sz;
   particle_ptr newpart;
   unsigned long int tics,last_tics;
   FLOAT temp;

   quit_altogether = FALSE;
   sx = xf-xs;
   sy = yf-ys;
   sz = zf-zs;

   /* create the particles */
   last_tics = clock();
   for (i=0;i<cnt;i++) {

      xpos = xs + sx*(rand()/(RAND_MAX+1.0));
      ypos = ys + sy*(rand()/(RAND_MAX+1.0));
      zpos = zs + sz*(rand()/(RAND_MAX+1.0));
      /* fprintf(stdout,"  testing pt %g %g %g\n",xpos,ypos,zpos); */

      /* this location is good, place the particle */
      newpart = new_particle(i,sim->new_part_mass,sim->new_part_rad,xpos,ypos,zpos,0.0,0.0,0.0);
      add_particle_to_cell(sim,newpart,top);
      /* fprintf(stdout,"  placed pt there\n"); */

      if (i/1000 != (i+1)/1000) {
         fprintf(stdout,".");
         fflush(stdout);
      }
   }
   fprintf(stdout,"\n");
   tics = clock();
   temp = ((FLOAT)tics-(FLOAT)last_tics)/CLOCKS_PER_SEC;

   if (i>cnt) fprintf(stdout,"Placed %d of %d particles\n",i-cnt-1,cnt);
   else fprintf(stdout,"Placed %d of %d particles\n",i,cnt);

   fprintf(stdout,"Took %g seconds of cpu time for it\n",temp);

   /* return 0 is all went well */
   return(0);
}


/*
 *  Create a cube/rectangle of non-intersecting particles
 */
int add_box_of_ni_particles(sim_ptr sim,cell_ptr top,int cnt,FLOAT xs,FLOAT xf,FLOAT ys,FLOAT yf,FLOAT zs,FLOAT zf) {

   int i,keep_trying,ntried,isitfree,quit_altogether;
   FLOAT xpos,ypos,zpos;
   FLOAT sx,sy,sz;
   particle_ptr newpart;

   quit_altogether = FALSE;
   sx = xf-xs;
   sy = yf-ys;
   sz = zf-zs;

   /* create the particles */
   for (i=0;i<cnt;i++) {

      keep_trying = TRUE;
      ntried = 0;

      /* fprintf(stdout,"placing particle %d\n",i+1); */

      /* try to find an open place for the particle */
      while (keep_trying) {

         ntried++;
         isitfree = TRUE;

         xpos = xs + sx*(rand()/(RAND_MAX+1.0));
         ypos = ys + sy*(rand()/(RAND_MAX+1.0));
         zpos = zs + sz*(rand()/(RAND_MAX+1.0));
         /* fprintf(stdout,"  testing pt %g %g %g\n",xpos,ypos,zpos); */

         /* is this location open? check walls, then other particles */
         if (xpos-xs < sim->new_part_rad) isitfree = FALSE;
         if (xs+sx-xpos < sim->new_part_rad) isitfree = FALSE;
         if (ypos-ys < sim->new_part_rad) isitfree = FALSE;
         if (ys+sy-ypos < sim->new_part_rad) isitfree = FALSE;
         if (zpos-zs < sim->new_part_rad) isitfree = FALSE;
         if (zs+sz-zpos < sim->new_part_rad) isitfree = FALSE;
         if (isitfree) isitfree = is_this_space_open2(top,xpos,ypos,zpos,sim->new_part_rad);
         if (isitfree) keep_trying = FALSE;

         /* have we just tried too many times already? */
         if (keep_trying && ntried > 2000) {
            keep_trying = FALSE;
            quit_altogether = TRUE;
         }
      }

      if (quit_altogether) {
         /* get me out of this shitbag loop */
         i += cnt;
      } else {
         /* this location is good, place the particle */
         newpart = new_particle(i,sim->new_part_mass,sim->new_part_rad,xpos,ypos,zpos,0.0,0.0,0.0);
         add_particle_to_cell(sim,newpart,top);
         /* fprintf(stdout,"  placed pt there\n"); */
      }

      if (i/1000 != (i+1)/1000) {
         fprintf(stdout,".");
         fflush(stdout);
      }
   }
   fprintf(stdout,"\n");

   if (i>cnt) fprintf(stdout,"Placed %d of %d particles\n",i-cnt-1,cnt);
   else fprintf(stdout,"Placed %d of %d particles\n",i,cnt);

   /* return 0 is all went well */
   return(0);
}


/*
 *  Create a sphere of randomly-placed particles
 */
int add_sphere_of_particles(sim_ptr sim,cell_ptr top,int cnt,FLOAT xc,FLOAT yc,FLOAT zc,FLOAT rad) {

   int i,keep_trying,ntried,isitfree,quit_altogether;
   FLOAT xpos,ypos,zpos;
   FLOAT xs,ys,zs,dist;
   particle_ptr newpart;

   quit_altogether = FALSE;
   xs = xc-rad;
   ys = yc-rad;
   zs = zc-rad;

   /* create the particles */
   for (i=0;i<cnt;i++) {

      keep_trying = TRUE;
      ntried = 0;

      // fprintf(stdout,"placing particle %d\n",i+1);

      /* try to find an open place for the particle */
      while (keep_trying) {

         ntried++;
         isitfree = TRUE;

         xpos = xs + 2.0*rad*(rand()/(RAND_MAX+1.0));
         ypos = ys + 2.0*rad*(rand()/(RAND_MAX+1.0));
         zpos = zs + 2.0*rad*(rand()/(RAND_MAX+1.0));
         // fprintf(stdout,"  testing pt %g %g %g\n",xpos,ypos,zpos);

         // is this space within the sphere?
         dist = sqrt(pow(xpos-xc,2)+pow(ypos-yc,2)+pow(zpos-zc,2));
         if (dist+sim->new_part_rad > rad) isitfree = FALSE;
         // fprintf(stdout,"    dist (%g) + newrad (%g) > rad (%g)\n",dist,sim->new_part_rad,rad);

         if (isitfree) keep_trying = FALSE;

         /* have we just tried too many times already? */
         if (keep_trying && ntried > 2000) {
            keep_trying = FALSE;
            quit_altogether = TRUE;
         }
      }

      if (quit_altogether) {
         /* get me out of this shitbag loop */
         i += cnt;
      } else {
         /* this location is good, place the particle */
         newpart = new_particle(i,sim->new_part_mass,sim->new_part_rad,xpos,ypos,zpos,0.0,0.0,0.0);
         add_particle_to_cell(sim,newpart,top);
         /* fprintf(stdout,"  placed pt there\n"); */
      }

      if (i/1000 != (i+1)/1000) {
         fprintf(stdout,".");
         fflush(stdout);
      }
   }
   fprintf(stdout,"\n");

   if (i>cnt) fprintf(stdout,"Placed %d of %d particles\n",i-cnt-1,cnt);
   else fprintf(stdout,"Placed %d of %d particles\n",i,cnt);

   /* return 0 is all went well */
   return(0);
}


/*
 *  Create a sphere of non-intersecting particles
 */
int add_sphere_of_ni_particles(sim_ptr sim,cell_ptr top,int cnt,FLOAT xc,FLOAT yc,FLOAT zc,FLOAT rad) {

   int i,keep_trying,ntried,isitfree,quit_altogether;
   FLOAT xpos,ypos,zpos;
   FLOAT xs,ys,zs,dist;
   particle_ptr newpart;

   quit_altogether = FALSE;
   xs = xc-rad;
   ys = yc-rad;
   zs = zc-rad;

   /* create the particles */
   for (i=0;i<cnt;i++) {

      keep_trying = TRUE;
      ntried = 0;

      // fprintf(stdout,"placing particle %d\n",i+1);

      /* try to find an open place for the particle */
      while (keep_trying) {

         ntried++;
         isitfree = TRUE;

         xpos = xs + 2.0*rad*(rand()/(RAND_MAX+1.0));
         ypos = ys + 2.0*rad*(rand()/(RAND_MAX+1.0));
         zpos = zs + 2.0*rad*(rand()/(RAND_MAX+1.0));
         // fprintf(stdout,"  testing pt %g %g %g\n",xpos,ypos,zpos);

         // is this space within the sphere?
         dist = sqrt(pow(xpos-xc,2)+pow(ypos-yc,2)+pow(zpos-zc,2));
         if (dist+sim->new_part_rad > rad) isitfree = FALSE;
         // fprintf(stdout,"    dist (%g) + newrad (%g) > rad (%g)\n",dist,sim->new_part_rad,rad);

         // is this location open?
         if (isitfree) isitfree = is_this_space_open2(top,xpos,ypos,zpos,sim->new_part_rad);
         if (isitfree) keep_trying = FALSE;

         /* have we just tried too many times already? */
         if (keep_trying && ntried > 2000) {
            keep_trying = FALSE;
            quit_altogether = TRUE;
         }
      }

      if (quit_altogether) {
         /* get me out of this shitbag loop */
         i += cnt;
      } else {
         /* this location is good, place the particle */
         newpart = new_particle(i,sim->new_part_mass,sim->new_part_rad,xpos,ypos,zpos,0.0,0.0,0.0);
         add_particle_to_cell(sim,newpart,top);
         /* fprintf(stdout,"  placed pt there\n"); */
      }

      if (i/1000 != (i+1)/1000) {
         fprintf(stdout,".");
         fflush(stdout);
      }
   }
   fprintf(stdout,"\n");

   if (i>cnt) fprintf(stdout,"Placed %d of %d particles\n",i-cnt-1,cnt);
   else fprintf(stdout,"Placed %d of %d particles\n",i,cnt);

   /* return 0 is all went well */
   return(0);
}


/*
 *  Check to see if a point in space can support a non-intersecting particle
 */
int is_this_space_open(cell_ptr cell,FLOAT x,FLOAT y,FLOAT z,FLOAT r){

   int i,j,k,isitfree;
   int passed_it_on = FALSE;
   FLOAT tr,dist;
   particle_ptr curr = cell->first;

   tr = 2.0*r;

   if (cell->level == 0) isitfree = TRUE;

   if (cell->has_subcells) {
      /* if the possible contact volume can be completely enclosed by one of the
       * subcells, then test that one instead */
      for (i=0;i<2;i++)
         if (x-tr > cell->s[i][0][0]->min[0] && x+tr < cell->s[i][0][0]->max[0])
            for (j=0;j<2;j++)
               if (y-tr > cell->s[0][j][0]->min[1] && y+tr < cell->s[0][j][0]->max[1])
                  for (k=0;k<2;k++)
                     if (z-tr > cell->s[0][0][k]->min[2] && z+tr < cell->s[0][0][k]->max[2]) {
                        isitfree = is_this_space_open(cell->s[i][j][k],x,y,z,r);
                        passed_it_on = TRUE;
                     }
   }

   if (!passed_it_on) {
      /* this means that we have to check against all particles in this cell
       * and its subcells */

      if (cell->has_subcells) {
         /* must check all of the particles in all of the subcells */
         for (i=0;i<2;i++)
            for (j=0;j<2;j++)
               for (k=0;k<2;k++) {
                  isitfree = is_this_space_open(cell->s[i][j][k],x,y,z,r);
                  if (!isitfree) return(FALSE);
               }
      } else {
         /* check the individual particles here */
         while (curr) {
            dist = sqrt(pow(x-curr->x[0][0],2)+pow(y-curr->x[0][1],2)+pow(z-curr->x[0][2],2));
            if (dist < tr) {
               /* the position is not free, stop right here */
               return(FALSE);
            } else {
               /* the verdict is still out, so keep looking */
               curr = curr->next;
            }
         }
      }
   }

   return(isitfree);
}


/*
 *  Check to see if a point in space can support a non-intersecting particle
 *
 *  This second method is clearly superior to the earlier method,
 *  there is no more banding of particles---the distribution is
 *  even. Now, to see if over time, the banding re-appears!
 */
int is_this_space_open2(cell_ptr cell,FLOAT x,FLOAT y,FLOAT z,FLOAT r){

   int i,j,k,isitfree;
   FLOAT tr,dist;
   particle_ptr curr;

   tr = 2.0*r;
   // if (cell->level == 0) isitfree = TRUE;

   // first, if it's outside of the possible contact bounds of
   //   this cube, then just return TRUE (meaning there's nothing
   //   in this cell to prevent the placement of the particle
   if (x < cell->min[0]-tr) return(TRUE);
   if (x > cell->max[0]+tr) return(TRUE);
   if (y < cell->min[1]-tr) return(TRUE);
   if (y > cell->max[1]+tr) return(TRUE);
   if (z < cell->min[2]-tr) return(TRUE);
   if (z > cell->max[2]+tr) return(TRUE);

   // then, if we have subcells, go check them all (yes, inefficient)

   if (cell->has_subcells) {
      // must check all of the subcells
      for (i=0;i<2;i++)
         for (j=0;j<2;j++)
            for (k=0;k<2;k++) {
               isitfree = is_this_space_open2(cell->s[i][j][k],x,y,z,r);
               // if any of these cells finds an obstruction, then we
               //   can abort the rest of the comparisons
               if (!isitfree) return(FALSE);
            }
   } else {
      // check vs. all of the particles in this cell
      curr = cell->first;
      while (curr) {
         dist = sqrt(pow(x-curr->x[0][0],2)+pow(y-curr->x[0][1],2)+pow(z-curr->x[0][2],2));
         if (dist < tr) {
            // the position is not free, stop right here
            return(FALSE);
         } else {
            // the verdict is still out, so keep looking
            curr = curr->next;
         }
      }
   }

   return(TRUE);
}


/*
 *  Create a cube/rectangle of non-intersecting particles
 *  instead of recieving dl from subroutine, recieve radius and other stuff
 */
// int add_strand_of_particles(sim_ptr sim,cell_ptr top,int strand_num,FLOAT dl,FLOAT xs,FLOAT xf,FLOAT ys,FLOAT yf,FLOAT zs,FLOAT zf) {
int add_strand_of_particles(sim_ptr sim,cell_ptr top,int strand_num,FLOAT rad,FLOAT dens,FLOAT E,FLOAT xs,FLOAT ys,FLOAT zs,FLOAT xf,FLOAT yf,FLOAT zf,FLOAT gx,FLOAT gy,FLOAT gz) {

   int i,num;
   FLOAT dl;
   FLOAT xpos,ypos,zpos;
   FLOAT sx,sy,sz;
   FLOAT tot_len,new_len;
   particle_ptr newpart,last_particle;
   strand_ptr this = &sim->strand[strand_num];

   // create and define the new strand
   this->index = strand_num;
   this->dens = dens;
   this->rad = rad;
   this->E = E;

   // fprintf(stdout,"recieved %lf %lf %lf  %lf %lf %lf\n",xs,ys,zs,xf,yf,zf);
   sx = xf-xs;
   sy = yf-ys;
   sz = zf-zs;
   tot_len = sqrt(sx*sx + sy*sy + sz*sz);
   // num is the number of segments, so num+1 is the number of particles to use
   // num = (int)(tot_len/dl);
   num = (int)(tot_len/(1.2*rad));
   this->nseg = num;
   dl = tot_len/num;
   this->dl = dl;
   fprintf(stdout,"Adding strand with length %g, %g and %d segments\n",tot_len,dl,num);

   // set longitudinal and bending stiffness
   this->k = this->E * rad*rad*M_PI / (num/tot_len);
   this->b = this->E * rad*rad*rad*rad*8.0*M_PI / 27.0;
   fprintf(stdout,"nseg %d  dl %g  k %g  b %g  mass %g\n",this->nseg,this->dl,this->k,this->b,dens*dl*rad*rad*M_PI);

   // now, this re-assignment of length is unneccessary
   // new_len = num*dl;
   // redefine sx,sy,sz to be the segment component lengths
   sx /= (FLOAT)num;
   sy /= (FLOAT)num;
   sz /= (FLOAT)num;

   // set the initial position
   xpos = xs;
   ypos = ys;
   zpos = zs;

   // now, the growth part
   if (isinf(gx)) {
      // it's not growing
      this->grows = FALSE;
   } else {
      this->grows = TRUE;
      fprintf(stdout,"   and it grows at %g %g %g\n",gx,gy,gz);
   }


   /* create the particles */

   // the first particle
   // newpart = new_particle(0,sim->new_part_mass,sim->new_part_rad,xpos,ypos,zpos,0.0,0.0,0.0);
   newpart = new_particle(0,dens*dl*rad*rad*M_PI,rad,xpos,ypos,zpos,0.0,0.0,0.0);
   add_particle_to_cell(sim,newpart,top);
   newpart->s_rad = dl/2.0;
   newpart->s_last = NULL;
   newpart->parent = this;
   last_particle = newpart;
   // sim->strand_start[strand_num] = newpart;
   // sim->strand[strand_num]->start = newpart;
   this->start = newpart;

   // add the middle particles
   for (i=1; i<num; i++) {
      // newpart = new_particle(i,sim->new_part_mass,sim->new_part_rad,
      //                        xpos+i*sx,ypos+i*sy,zpos+i*sz,0.0,0.0,0.0);
      newpart = new_particle(i,dens*dl*rad*rad*M_PI,rad,
                             xpos+i*sx,ypos+i*sy,zpos+i*sz,0.0,0.0,0.0);
      add_particle_to_cell(sim,newpart,top);
      newpart->s_rad = dl/2.0;
      newpart->s_last = last_particle;
      newpart->parent = this;
      last_particle->s_next = newpart;
      last_particle = newpart;
   }

   // add the last particle
   // newpart = new_particle(num,sim->new_part_mass,sim->new_part_rad,
   //                        xpos+num*sx,ypos+num*sy,zpos+num*sz,0.0,0.0,0.0);
   newpart = new_particle(num,dens*dl*rad*rad*M_PI,rad,
                          xpos+num*sx,ypos+num*sy,zpos+num*sz,0.0,0.0,0.0);
   add_particle_to_cell(sim,newpart,top);
   newpart->s_rad = dl/2.0;
   newpart->s_last = last_particle;
   last_particle->s_next = newpart;
   newpart->s_next = NULL;
   newpart->parent = this;

   // make appropriate modifications if this is a growing strand
   if (this->grows) {
      // first particle is stationary
      this->start->stationary = TRUE;
      // next part is, too
      this->start->s_next->stationary = TRUE;
   }

   // if all went well, return 0
   return (0);
}


/*
 *  Create a galaxy with cnt particles
 *  call with
 *  add_galaxy(sim,top,cnt,x,y,z,nx,ny,nz,rad,vx,vy,vz);
 */
int add_galaxy(sim_ptr sim,cell_ptr top,int cnt,FLOAT x,FLOAT y,FLOAT z,FLOAT nx,FLOAT ny,FLOAT nz,FLOAT rad,FLOAT vx,FLOAT vy,FLOAT vz){

   int i;
   /* v[2][0:2] is the basis vector for the rotational axis of the galaxy */
   FLOAT v[3][3];
   FLOAT len;
   FLOAT mpp = 1.0/cnt;
   FLOAT theta,r,n;
   FLOAT xpos,ypos,zpos,xvel,yvel,zvel;
   particle_ptr newpart;

   /* normalize the vector normal to the plane of the galaxy */
   len = sqrt(nx*nx+ny*ny+nz*nz);
   v[2][0] = nx/len;
   v[2][1] = ny/len;
   v[2][2] = nz/len;

   /* create the basis vectors v[0:2][] */

   /* create the particles */
   for (i=0;i<cnt;i++) {

      /* theta is nicely random */
      theta = 2.0*M_PI*(rand()/(RAND_MAX+1.0));

      /* determine r using exponential power law */
      /* but not yet! */
      r = rad*(rand()/(RAND_MAX+1.0));

      /* and determine n, the z component, by a power law, too */
      /* but not yet! */
      n = 0.1*rad*(rand()/(RAND_MAX+1.0));

      /* calculate particle initial location */
      xpos = x + r*cos(theta)*v[0][0] + r*sin(theta)*v[1][0] + n*v[2][0];
      ypos = y + r*cos(theta)*v[0][1] + r*sin(theta)*v[1][1] + n*v[2][1];
      zpos = z + r*cos(theta)*v[0][2] + r*sin(theta)*v[1][2] + n*v[2][2];

      /* determine particle's orbital speed */

      /* calculate paricle's initial velocity */
      xvel = vx;
      yvel = vy;
      zvel = vz;

      newpart = new_particle(i,mpp,sim->new_part_rad,xpos,ypos,zpos,xvel,yvel,zvel);
      add_particle_to_cell(sim,newpart,top);
   }

   /* return 0 is all went well */
   return(0);
}


/*
 *  Create a new particle
 */
particle_ptr new_particle(int i,FLOAT m,FLOAT r,FLOAT x,FLOAT y,FLOAT z,FLOAT u,FLOAT v,FLOAT w){

   particle_ptr newpart;

   newpart = (PARTICLE*)malloc(sizeof(PARTICLE));
   newpart->index = i;
   newpart->mass = m;
   newpart->rad = r;
   newpart->s_rad = r;
   newpart->x[0][0] = x;
   newpart->x[0][1] = y;
   newpart->x[0][2] = z;
   newpart->u[0][0] = u;
   newpart->u[0][1] = v;
   newpart->u[0][2] = w;
   newpart->a[0] = 0.0;
   newpart->a[1] = 0.0;
   newpart->a[2] = 0.0;
   /* 
   newpart->x[1][0] = 0.0;
   newpart->x[1][1] = 0.0;
   newpart->x[1][2] = 0.0;
   newpart->u[1][0] = 0.0;
   newpart->u[1][1] = 0.0;
   newpart->u[1][2] = 0.0;
   */
   newpart->stationary = FALSE;
   newpart->flag = FALSE;
   newpart->s_last = NULL;
   newpart->s_next = NULL;
   newpart->next = NULL;
   newpart->parent = NULL;

   // fprintf(stderr,"Added particle %d with mass %g and radius %g\n",newpart->index,newpart->mass,newpart->rad);

   return(newpart);
}


/*
 *  Create a new stationary particle
 */
particle_ptr new_stationary_particle(int i,FLOAT r,FLOAT x,FLOAT y,FLOAT z){

   particle_ptr newpart;

   newpart = (PARTICLE*)malloc(sizeof(PARTICLE));
   newpart->index = i;
   newpart->mass = 0.0;
   newpart->rad = r;
   newpart->s_rad = r;
   newpart->x[0][0] = x;
   newpart->x[0][1] = y;
   newpart->x[0][2] = z;
   newpart->u[0][0] = 0.0;
   newpart->u[0][1] = 0.0;
   newpart->u[0][2] = 0.0;
   newpart->a[0] = 0.0;
   newpart->a[1] = 0.0;
   newpart->a[2] = 0.0;
   newpart->stationary = TRUE;
   newpart->flag = FALSE;
   newpart->next = NULL;
   newpart->s_last = NULL;
   newpart->s_next = NULL;
   newpart->parent = NULL;

   return(newpart);
}


/*
 * Read the input file and set the listed parameters
 */
int read_input_file(fileprop_ptr file,sim_ptr sim,cell_ptr top) {

   char token[13][32];
   char twochar[2];
   char sbuf[512];
   FILE *infile;

   /* open file for reading */
   infile = fopen(file->input_fn,"r");
   if (infile==NULL) {
      fprintf(stderr,"Could not open input file %s\n",file->input_fn);
      fflush(stderr);
      return(1);
   }
   fprintf(stdout,"Opening file %s\n",file->input_fn);
   fflush(stdout);

   /* read a line from the input file */
   while (fscanf(infile,"%[^\n]",sbuf) != EOF) {

      // fprintf(stdout,"%s\n",sbuf);

      /* grab the first word */
      sscanf(sbuf,"%s %s %s %s %s %s %s %s %s %s %s %s %s",token[0],token[1],token[2],token[3],token[4],token[5],token[6],token[7],token[8],token[9],token[10],token[11],token[12]);
      /* fprintf(stdout,"   first token is %s\n",token[0]); */

      if (token[0][0] == '#') {
         /* read a comment line, or some other descriptor */
         fscanf(infile,"%[\n]",twochar);    /* read up to newline */
         /* fprintf(stdout,"%s\n",sbuf);        /* write comment */

      } else {
         /* first word is a parameter name */
         /* read the parameter value */
         /* fprintf(stdout,"   second token is %s\n",token[1]); */

         if (strncmp(token[0],"outfile_root",12) == 0) {
            strcpy(file->out_fn_root,token[1]);
         } else if (strncmp(token[0],"particles",9) == 0) {
            sim->particle_cnt = atoi(token[1]);
         } else if (strncmp(token[0],"write_pgm",9) == 0) {
            if (strncmp(token[1],"yes",1) == 0)
               file->write_pgm = TRUE;
            else
               file->write_pgm = FALSE;
         } else if (strncmp(token[0],"write_gif",9) == 0) {
            if (strncmp(token[1],"yes",1) == 0)
               file->write_gif = TRUE;
            else
               file->write_gif = FALSE;
         } else if (strncmp(token[0],"write_png",9) == 0) {
            if (strncmp(token[1],"yes",1) == 0)
               file->write_png = TRUE;
            else
               file->write_png = FALSE;
         } else if (strncmp(token[0],"write_dot",9) == 0) {
            if (strncmp(token[1],"yes",1) == 0)
               file->write_dot = TRUE;
            else
               file->write_dot = FALSE;
         } else if (strncmp(token[0],"write_rad",9) == 0) {
            if (strncmp(token[1],"yes",1) == 0)
               file->write_rad = TRUE;
            else
               file->write_rad = FALSE;
         } else if (strncmp(token[0],"write_part",10) == 0) {
            if (strncmp(token[1],"yes",1) == 0)
               file->write_part = TRUE;
            else
               file->write_part = FALSE;
         } else if (strncmp(token[0],"end_time",8) == 0) {
            sim->end_time = atof(token[1]);
         } else if (strncmp(token[0],"dt",2) == 0) {
            sim->dt = atof(token[1]);
         } else if (strncmp(token[0],"output_dt",9) == 0) {
            sim->output_dt = atof(token[1]);
         } else if (strncmp(token[0],"max_levels",10) == 0) {
            sim->max_levels = atoi(token[1]);
         } else if (strncmp(token[0],"max_ppc",7) == 0) {
            sim->max_parts_in_cell = atoi(token[1]);
         } else if (strncmp(token[0],"G",1) == 0) {
            sim->G = atof(token[1]);
         } else if (strncmp(token[0],"delta",5) == 0) {
            sim->delta = atof(token[1]);
         } else if (strncmp(token[0],"comp_domain",11) == 0) {
            /* set initial size of first cell */
            top->min[0] = atof(token[1]);
            top->max[0] = atof(token[2]);
            top->min[1] = atof(token[3]);
            top->max[1] = atof(token[4]);
            top->min[2] = atof(token[5]);
            top->max[2] = atof(token[6]);
         } else if (strncmp(token[0],"boundary",8) == 0) {
            if (strncmp(token[1],"wall",4) == 0) {
               sim->bdry[0][0] = WALL;
               sim->bdry[0][1] = WALL;
               sim->bdry[1][0] = WALL;
               sim->bdry[1][1] = WALL;
               sim->bdry[2][0] = WALL;
               sim->bdry[2][1] = WALL;
            } else if (strncmp(token[1],"open",4) == 0) {
               sim->bdry[0][0] = OPEN;
               sim->bdry[0][1] = OPEN;
               sim->bdry[1][0] = OPEN;
               sim->bdry[1][1] = OPEN;
               sim->bdry[2][0] = OPEN;
               sim->bdry[2][1] = OPEN;
            }
         } else if (strncmp(token[0],"image_size",10) == 0) {
            file->out_img_size = atoi(token[1]);
            file->out->n[0] = atoi(token[1]);
            file->out->n[1] = atoi(token[1]);
            file->out->n[2] = atoi(token[1]);
            // fprintf(stdout,"img size is %d\n",file->out_img_size);
         } else if (strncmp(token[0],"use_self_grav",13) == 0) {
            if (strncmp(token[1],"yes",1) == 0)
               sim->use_self_grav = TRUE;
            else
               sim->use_self_grav = FALSE;
         } else if (strncmp(token[0],"use_uniform_grav",16) == 0) {
            if (strncmp(token[1],"yes",1) == 0)
               sim->use_uniform_grav = TRUE;
            else
               sim->use_uniform_grav = FALSE;
         } else if (strncmp(token[0],"use_contact",11) == 0) {
            if (strncmp(token[1],"yes",1) == 0)
               sim->use_contact = TRUE;
            else
               sim->use_contact = FALSE;
         } else if (strncmp(token[0],"g",1) == 0) {
            sim->g[0] = atof(token[1]);
            sim->g[1] = atof(token[2]);
            sim->g[2] = atof(token[3]);
         } else if (strncmp(token[0],"rk",2) == 0) {
            sim->rk = atof(token[1]);
         } else if (strncmp(token[0],"rc",2) == 0) {
            sim->rc = atof(token[1]);
         } else if (strncmp(token[0],"rad",3) == 0) {
            sim->new_part_rad = atof(token[1]);
         } else if (strncmp(token[0],"mass",4) == 0) {
            sim->new_part_mass = atof(token[1]);
         } else if (strncmp(token[0],"block",5) == 0) {
            sim->block[sim->num_blocks][0] = atof(token[1]);
            sim->block[sim->num_blocks][1] = atof(token[2]);
            sim->block[sim->num_blocks][2] = atof(token[3]);
            sim->block[sim->num_blocks][3] = atof(token[4]);
            sim->block[sim->num_blocks][4] = atof(token[5]);
            sim->block[sim->num_blocks][5] = atof(token[6]);
            sim->block[sim->num_blocks][6] = atof(token[7]);
            sim->num_blocks++;
         } else if (strncmp(token[0],"niblock",7) == 0) {
            sim->niblock[sim->num_niblocks][0] = atof(token[1]);
            sim->niblock[sim->num_niblocks][1] = atof(token[2]);
            sim->niblock[sim->num_niblocks][2] = atof(token[3]);
            sim->niblock[sim->num_niblocks][3] = atof(token[4]);
            sim->niblock[sim->num_niblocks][4] = atof(token[5]);
            sim->niblock[sim->num_niblocks][5] = atof(token[6]);
            sim->niblock[sim->num_niblocks][6] = atof(token[7]);
            sim->num_niblocks++;
         } else if (strncmp(token[0],"sphere",6) == 0) {
            sim->sphere[sim->num_spheres][0] = atof(token[1]);
            sim->sphere[sim->num_spheres][1] = atof(token[2]);
            sim->sphere[sim->num_spheres][2] = atof(token[3]);
            sim->sphere[sim->num_spheres][3] = atof(token[4]);
            sim->sphere[sim->num_spheres][4] = atof(token[5]);
            sim->num_spheres++;
         } else if (strncmp(token[0],"nisphere",8) == 0) {
            sim->nisphere[sim->num_nispheres][0] = atof(token[1]);
            sim->nisphere[sim->num_nispheres][1] = atof(token[2]);
            sim->nisphere[sim->num_nispheres][2] = atof(token[3]);
            sim->nisphere[sim->num_nispheres][3] = atof(token[4]);
            sim->nisphere[sim->num_nispheres][4] = atof(token[5]);
            sim->num_nispheres++;
         } else if (strncmp(token[0],"add_part",8) == 0) {
            sim->indiv_part[sim->num_indiv_parts][0] = atof(token[1]);
            sim->indiv_part[sim->num_indiv_parts][1] = atof(token[2]);
            sim->indiv_part[sim->num_indiv_parts][2] = atof(token[3]);
            sim->indiv_part[sim->num_indiv_parts][3] = atof(token[4]);
            sim->indiv_part[sim->num_indiv_parts][4] = atof(token[5]);
            sim->indiv_part[sim->num_indiv_parts][5] = atof(token[6]);
            sim->indiv_part[sim->num_indiv_parts][6] = atof(token[7]);
            sim->num_indiv_parts++;
         } else if (strncmp(token[0],"read_part",9) == 0) {
            strcpy(sim->read_part[sim->num_read_part],token[1]);
            sim->part_xform[sim->num_read_part][0] = atof(token[2]);
            sim->part_xform[sim->num_read_part][1] = atof(token[3]);
            sim->part_xform[sim->num_read_part][2] = atof(token[4]);
            sim->part_xform[sim->num_read_part][3] = atof(token[5]);
            sim->part_xform[sim->num_read_part][4] = atof(token[6]);
            sim->part_xform[sim->num_read_part][5] = atof(token[7]);
            sim->num_read_part++;
         } else if (strncmp(token[0],"read_stat",9) == 0) {
            strcpy(sim->read_stat[sim->num_read_stat],token[1]);
            sim->stat_xform[sim->num_read_stat][0] = atof(token[2]);
            sim->stat_xform[sim->num_read_stat][1] = atof(token[3]);
            sim->stat_xform[sim->num_read_stat][2] = atof(token[4]);
            sim->num_read_stat++;
         } else if (strncmp(token[0],"density",7) == 0) {
            sim->ff2->n[0] = atoi(token[1]);
            sim->ff2->n[1] = atoi(token[1]);
            sim->ff2->n[2] = atoi(token[1]);
            sim->ff3->n[0] = atoi(token[1]);
            sim->ff3->n[1] = atoi(token[1]);
            sim->ff3->n[2] = atoi(token[1]);
            sim->use_density_field = TRUE;
         } else if (strncmp(token[0],"strand",6) == 0) {
            sim->block[sim->num_strands][0] = atof(token[1]);
            sim->block[sim->num_strands][1] = atof(token[2]);
            sim->block[sim->num_strands][2] = atof(token[3]);
            sim->block[sim->num_strands][3] = atof(token[4]);
            sim->block[sim->num_strands][4] = atof(token[5]);
            sim->block[sim->num_strands][5] = atof(token[6]);
            sim->block[sim->num_strands][6] = atof(token[7]);
            sim->block[sim->num_strands][7] = atof(token[8]);
            sim->block[sim->num_strands][8] = atof(token[9]);
            // and more for growing strands!
            fprintf(stderr,"strlen(%s) is %d\n",token[10],strlen(token[10]));
            if (strlen(token[10]) > 1) {
               sim->block[sim->num_strands][9] = atof(token[10]);
               sim->block[sim->num_strands][10] = atof(token[11]);
               sim->block[sim->num_strands][11] = atof(token[12]);
            } else {
               sim->block[sim->num_strands][9] = 1./0.;
            }
            sim->num_strands++;
         }

         fscanf(infile,"%[\n]",twochar);    /* read newline */
      }
   }

   fclose(infile);

   /* ret_val is 0 is file is OK and read, 1 if not */
   return(0);
}


/*
 * Parse the command-line arguments
 */
int parse_args(int argc,char **argv,fileprop_ptr file) {

   int i,some_logical;
   FLOAT some_float;

   (void) strcpy(file->exectuable_fn,argv[0]);
   if (argc < 2) (void) Usage(file->exectuable_fn,0);
   if (strncmp(argv[1], "-help", 2) == 0)
      (void) Usage(file->exectuable_fn,0);
   if (strncmp(argv[1], "help", 4) == 0)
      (void) Usage(file->exectuable_fn,0);
   (void) strcpy(file->input_fn,argv[1]);
   for (i=2; i<argc; i++) {
      (void) Usage(file->exectuable_fn,0);
   }

   return(0);
}


/*
 * This function writes basic usage information to stderr,
 * and then quits. Too bad.
 */
int Usage(char progname[80],int status) {

   /* Usage for grav3d */
   static char **cpp, *help_message[] =
   {
       "  The input file should be a raw ASCII file containing 2 columns, the first",
       "  with a parameter name (dt,particles,G,write_rad) and the second with either",
       "  an integer or floating-point value or with a \'yes\' or \'no\'.",
       " ",
       "  There is running output to stdout, but PGM and Radiance data are written",
       "  to disk.",
       " ",
       NULL
   };

   fprintf(stderr,"\n  Usage:  %s infile\n\n", progname);
   for (cpp = help_message; *cpp; cpp++)
      fprintf(stderr, "%s\n", *cpp);
      fflush(stderr);
   exit(status);
   return(0);
}


/*
 * allocate memory for a three-dimensional array of FLOATs
 */
FLOAT*** allocate_3d_array_F(int nx, int ny, int nz) {

   int i,j;
   FLOAT ***array = (FLOAT ***)malloc(nx * sizeof(FLOAT **));

   array[0] = (FLOAT **)malloc(nx * ny * sizeof(FLOAT *));
   array[0][0] = (FLOAT *)malloc(nx * ny * nz * sizeof(FLOAT));

   for (i=1; i<nx; i++)
      array[i] = array[0] + i * ny;

   for (i=0; i<nx; i++) {
      if (i!=0)
         array[i][0] = array[0][0] + i * ny * nz;
      for (j=1; j<ny; j++)
         array[i][j] = array[i][0] + j * nz;
   }

   return(array);
}


/*
 * allocate memory for a two-dimensional array of FLOATs
 */
FLOAT** allocate_2d_array_F(int nx, int ny) {

   int i,j;
   FLOAT **array = (FLOAT **)malloc(nx * sizeof(FLOAT *));

   array[0] = (FLOAT *)malloc(nx * ny * sizeof(FLOAT));

   for (i=1; i<nx; i++)
      array[i] = array[0] + i * ny;

   return(array);
}
